/*
 * ADOBE CONFIDENTIAL
 *
 * Copyright (c) 2015 Adobe Systems Incorporated. All rights reserved.
 *
 * NOTICE:  All information contained herein is, and remains
 * the property of Adobe Systems Incorporated and its suppliers,
 * if any.  The intellectual and technical concepts contained
 * herein are proprietary to Adobe Systems Incorporated and its
 * suppliers and are protected by trade secret or copyright law.
 * Dissemination of this information or reproduction of this material
 * is strictly forbidden unless prior written permission is obtained
 * from Adobe Systems Incorporated.
 */

/*jslint vars: true, node: true, plusplus: true, devel: true, nomen: true, indent: 4, expr:true */
/*global beforeEach: false, afterEach: false, it: false, describe: false, expect: false*/

"use strict";

var sinon = require("sinon"),
    Q = require("q"),
    Path = require("path"),
    PathUtils = require("shared/PathUtils"),
    PSDialogs = require("../lib/ps-dialogs"),
    StringUtils = require("shared/StringUtils");

describe("PSDialogs", function () {
    var sandbox = sinon.sandbox.create(),
        // Inputs
        folder = "/Users/Batman/Desktop",
        basename = "batman",
        ext = "jpg",
        documentBasename = "gotham-skyline",
        // Other inputs
        folder2 = "/Users/Robin/Desktop",
        basename2 = "robin",
        documentBasename2 = "batcave-panorama",
        // File paths
        unresolvedUserChosenFilePath = "~/Creative Cloud Files/user-chosen-file-name.jpg",
        userChosenFilePath = "/Users/Batman/Creative Cloud Files/user-chosen-file-name.jpg",
        userChosenFilePathFolder = Path.dirname(userChosenFilePath),
        unresolvedExtensionlessUserChosenFilePath = "~/Creative Cloud Files/user-chosen-file-name",
        extensionlessUserChosenFilePath = "/Users/Batman/Creative Cloud Files/user-chosen-file-name",
        expectedFileDialogJSX = 'var fileObj = File("/Users/Batman/Desktop/batman.jpg").saveDlg(); fileObj ? fileObj.fsName : "";',
        // Folder paths
        userChosenFolder = "/Users/Batman/Creative Cloud Files/user-chosen-folder",
        unresolvedUserChosenFolder = "~/Creative Cloud Files/user-chosen-folder",
        expectedFolderDialogJSX = 'var folderObj = Folder("/Users/Batman/Desktop").selectDlg(); folderObj ? folderObj.fsName : "";',
        // Paths related to escaping
        unescapedFolderForJSX = "/\"/'/\\/\"/'/\\",
        escapedFolderForJSX = "/\\\"/\\'/\\\\/\\\"/\\'/\\\\",
        unescapedBasenameForJSX = "bat\\man",
        escapedBasenameForJSX = "bat\\\\man",
        unescapedFilePathForJSX = unescapedFolderForJSX + "/" + unescapedBasenameForJSX + ".jpg",
        escapedFilePathForJSX =  escapedFolderForJSX + "/" + escapedBasenameForJSX + ".jpg",
        expectedEscapedFileDialogJSX = 'var fileObj = File("' + escapedFilePathForJSX  + '").saveDlg(); fileObj ? fileObj.fsName : "";',
        expectedEscapedFolderDialogJSX = 'var folderObj = Folder("' + escapedFolderForJSX + '").selectDlg(); folderObj ? folderObj.fsName : "";';

    afterEach(function () {
        sandbox.restore();
    });

    describe("alert", function () {
        it("runs JSX to bring up an alert", function() {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve())
            });

            return expect(psDialogs.alert("hello world")).to.eventually.be.fulfilled
                .then(function () {
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce
                        .and.calledWith('alert("hello world", " ");');
                });
        });

        it("escapes the alert message for the JSX string", function() {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve())
            });

            return expect(psDialogs.alert("\"hello\\world\'")).to.eventually.be.fulfilled
                .then(function () {
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce
                        .and.calledWith('alert("\\"hello\\\\world\\\'", " ");');
                });
        });
    }); // alerts

    describe("promptForFile", function () {
        it("runs JSX to bring up a dialog", function () {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve(userChosenFilePath))
            });

            return expect(psDialogs.promptForFile(folder, basename, ext)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result).to.equal(userChosenFilePath);
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedFileDialogJSX);
                });
        });

        it("escapes paths for JSX strings", function () {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve(userChosenFilePath))
            });

            sandbox.spy(StringUtils, "escapeForCodeContext");

            return expect(psDialogs.promptForFile(unescapedFolderForJSX, unescapedBasenameForJSX, ext)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(StringUtils.escapeForCodeContext).to.have.been.calledOnce.and.calledWith(unescapedFilePathForJSX);
                    expect(result).to.equal(userChosenFilePath);
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedEscapedFileDialogJSX);
                });
        });

        it("throws an error with a message of 'cancel' if the user cancels the dialog", function () {
            var psDialogs = new PSDialogs({
                // The JSX result is undefined when the user cancels the dialog.
                evaluateJSXString: sinon.stub().returns(Q.resolve(undefined))
            });

            return expect(psDialogs.promptForFile(folder, basename, ext)).to.eventually.be.rejected
                .then(function (e) {
                    expect(e instanceof Error).to.be.true;
                    expect(e.message).to.equal("cancel");
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedFileDialogJSX);
                });
        });

        it("resolves tildes and URI-encoded characters in user chosen paths", function () {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve(unresolvedUserChosenFilePath))
            });

            sandbox.stub(PathUtils, "getHomePath").returns("/Users/Batman");

            return expect(psDialogs.promptForFile(folder, basename, ext)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result).to.equal(userChosenFilePath);
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedFileDialogJSX);
                });
        });

        it("calls the appropriate function to add a file extension if needed", function () {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve(unresolvedExtensionlessUserChosenFilePath))
            });

            sandbox.stub(PathUtils, "getHomePath").returns("/Users/Batman");
            sandbox.spy(PathUtils, "addExtIfNeededWithoutCausingConflicts");

            return expect(psDialogs.promptForFile(folder, basename, ext)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result).to.equal(userChosenFilePath);
                    expect(PathUtils.addExtIfNeededWithoutCausingConflicts).to.have.been.calledOnce.and.calledWith(extensionlessUserChosenFilePath);
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedFileDialogJSX);
                });
        });
    }); // promptForFile

    describe("promptForFolder", function () {
        it("runs JSX to bring up a dialog", function () {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve(userChosenFolder))
            });

            return expect(psDialogs.promptForFolder(folder)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result).to.equal(userChosenFolder);
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedFolderDialogJSX);
                });
        });

        it("escapes paths for JSX strings", function () {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve(userChosenFilePath))
            });

            sandbox.spy(StringUtils, "escapeForCodeContext");

            return expect(psDialogs.promptForFolder(unescapedFolderForJSX)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(StringUtils.escapeForCodeContext).to.have.been.calledOnce.and.calledWith(unescapedFolderForJSX);
                    expect(result).to.equal(userChosenFilePath);
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedEscapedFolderDialogJSX);
                });
        });

        it("throws an error with a message of 'cancel' if the user cancels the dialog", function () {
            var psDialogs = new PSDialogs({
                // The JSX result is undefined when the user cancels the dialog.
                evaluateJSXString: sinon.stub().returns(Q.resolve(undefined))
            });

            return expect(psDialogs.promptForFolder(folder)).to.eventually.be.rejected
                .then(function (e) {
                    expect(e instanceof Error).to.be.true;
                    expect(e.message).to.equal("cancel");
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedFolderDialogJSX);
                });
        });

        it("resolves tildes and URI-encoded characters in user chosen paths", function () {
            var psDialogs = new PSDialogs({
                evaluateJSXString: sinon.stub().returns(Q.resolve(unresolvedUserChosenFolder))
            });

            sandbox.stub(PathUtils, "getHomePath").returns("/Users/Batman");

            return expect(psDialogs.promptForFolder(folder)).to.eventually.be.fulfilled
                .then(function (result) {
                    expect(result).to.equal(userChosenFolder);
                    expect(psDialogs._generator.evaluateJSXString).to.have.been.calledOnce.and.calledWith(expectedFolderDialogJSX);
                });
        });
    }); // promptForFolder
});
